﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using WizlinkPlugin.Enums;

namespace WizlinkPlugin
{
    /// <summary>
    /// Build the project to receive the wll library that would be the entry to your Plugin. 
    /// If you use any nuget packages or external libraries, remember to copy them 
    /// together with your plugin!
    /// .pdb files can be omited.
    /// </summary>
    public class HelloWizlinkPlugin : PluginBase
    {
        #region Simple, short running Method
        /// <summary>
        /// This is a method that accepts a string as an input parameter, prepends some text to it 
        /// and returns it as an output parameter
        /// </summary>
        /// <param name="name">in parameter</param>
        /// <returns></returns>
        public string HelloWizlink(string name)
        {
            return $"Wizlink says hello to {name}";
        }
        #endregion

        #region Long, time-consuming operation 
        /// <summary>
        /// Recommended way of performing time-consuming methods.
        /// You can also use the Description attribute to present additional data in Wizlink
        /// </summary>
        /// <param name="delay"></param>
        /// <param name="cancellationToken">Token passed for plugin to respond to workflow cancellation</param>
        /// <returns></returns>
        [Description("Returns information about the duration of the method. The parameter defines the duration of the method in seconds")]
        public async Task<string> DoAsync(int delay, CancellationToken cancellationToken = default)
        {
            await DelayAsync(delay, cancellationToken);
            return $"The method was performed for {delay} seconds.";
        }

        private async Task DelayAsync(int delay, CancellationToken cancellationToken = default)
        {
            await Task.Delay(TimeSpan.FromSeconds(delay), cancellationToken);
        }
        #endregion

        #region Multiple outputs
        /// <summary>
        /// Method that outputs two values
        /// </summary>
        /// <param name="a"></param>
        /// <param name="b"></param>
        /// <param name="additionResult"></param>
        /// <param name="multiplicationResult"></param>
        public void AddAndMultiply(int a, int b, out int additionResult, out int multiplicationResult)
        {
            additionResult = a + b;
            multiplicationResult = a * b;
        }
        #endregion

        #region Logging
        /// <summary>
        /// Tests the operation of logging in real time.
        /// </summary>
        /// <param name="counter">Number of iterations.</param>
        /// <param name="cancellationToken">Token passed for plugin to respond to workflow cancellation</param>
        /// <returns></returns>
        public async Task LoggerTest(int counter, CancellationToken cancellationToken)
        {
            Log("Example Error Log with Exception", LogLevel.Error, new Exception("Hello Exception!"));

            for (int i = 0; i < counter; i++)
            {
                Log($"Example Default Log: iteration {i + 1}");
                await Task.Delay(TimeSpan.FromSeconds(0.5), cancellationToken);
            }

            Log("Example Warning Log", LogLevel.Warning);
        }
        #endregion
    }
}
